/* MAXFLOW, Problema del Máximo Flujo */

/* Escrito en GNU MathProg por Andrew Makhorin <mao@gnu.org> */

/* El problema del máximo flujo en una red G = (V, E), en la que V es
   un conjunto de nodos y E es un conjunto de arcos que es un subconjunto
   de V x V, busca maximizar el flujo desde un nodo s (fuente) dado hasta
   otro nodo t (sumidero) dado, sujeto a restricciones de conservación de
   los flujos en cada nodo y a las capacidades de cada arco. */

param n, integer, >= 2;
/* número de nodos */

set V, default {1..n};
/* conjunto de nodos */

set E, within V cross V;
/* conjunto de arcos */

param a{(i,j) in E}, > 0;
/* a[i,j] es la capacidad del arco (i,j) */

param s, symbolic, in V, default 1;
/* nodo fuente */

param t, symbolic, in V, != s, default n;
/* nodo sumidero */

var x{(i,j) in E}, >= 0, <= a[i,j];
/* x[i,j] es el flujo elemental a través del arco (i,j) a ser encontrado */

var flow, >= 0;
/* flujo total desde s hasta t */

s.t. node{i in V}:
/* node[i] is conservation constraint for node i */

  sum{(j,i) in E} x[j,i] + (if i = s then flow)
  /* flujo consolidado hacia el nodo i a través de todos los arcos entrantes */

  = /* debe ser igual a */

  sum{(i,j) in E} x[i,j] + (if i = t then flow);
  /* flujo consolidado desde el nodo i a través de todos los arcos salientes */

maximize obj: flow;
/* el objetivo es maximizar el flujo total a través de la red */

solve;

printf{1..56} "="; printf "\n";
printf "El Flujo Máximo desde el nodo %s hasta el nodo %s es %g\n\n", s, t, flow;
printf "Nodo inicial   Nodo final   Capacidad arco   Flujo arco\n";
printf "------------   ----------   --------------   ----------\n";
printf{(i,j) in E: x[i,j] != 0}: "%12s   %10s   %14g   %10g\n", i, j,
   a[i,j], x[i,j];
printf{1..56} "="; printf "\n";

data;

/* Estos datos corresponden al ejemplo de la presentación */

/* La solución óptima es 13 */

param n := 7;

param : E :   a :=
       1 2    5
       1 3    4
       1 4    4
       2 5    2
       2 7    6
       3 5    6
       3 4    4
       4 3    4
       4 6    5
       6 5    3
       6 7    2
       5 7    8 ;

end;
